<?php
// api/login.php - Handle login/signup
header('Content-Type: application/json');
require_once '../config.php';

$data = json_decode(file_get_contents('php://input'), true);
$email = filter_var(trim($data['email'] ?? ''), FILTER_SANITIZE_EMAIL);
$password = $data['password'] ?? '';

if (!filter_var($email, FILTER_VALIDATE_EMAIL) || empty($password)) {
    http_response_code(400);
    echo json_encode(['error' => 'Valid email and password required']);
    exit;
}

$pdo = getDB();
checkRateLimit($pdo, 0, 'login'); // Rate limit unauthenticated

$stmt = $pdo->prepare("SELECT id, password FROM users WHERE email = :email");
$stmt->execute(['email' => $email]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

if ($user) {
    // Login
    if (password_verify($password, $user['password'])) {
        $jwt = generateJWT($user['id']);
        $stmt = $pdo->prepare("UPDATE users SET token = :token, last_login = NOW() WHERE id = :id");
        $stmt->execute(['token' => $jwt, 'id' => $user['id']]);
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'message' => 'Login successful',
            'user_id' => $user['id'],
            'token' => $jwt
        ]);
    } else {
        http_response_code(401);
        echo json_encode(['error' => 'Invalid password']);
    }
} else {
    // Signup
    $hashed_pass = password_hash($password, PASSWORD_DEFAULT);
    $stmt = $pdo->prepare("INSERT INTO users (email, password, created_at) VALUES (:email, :password, NOW())");
    try {
        $stmt->execute(['email' => $email, 'password' => $hashed_pass]);
        $user_id = $pdo->lastInsertId();
        $jwt = generateJWT($user_id);
        $stmt = $pdo->prepare("UPDATE users SET token = :token WHERE id = :id");
        $stmt->execute(['token' => $jwt, 'id' => $user_id]);

        // Create user directory
        $user_dir = USER_DATA_DIR . $email . '/';
        if (!file_exists($user_dir)) mkdir($user_dir, 0755, true);
        foreach (['contacts', 'call_logs', 'messages', 'images', 'videos', 'documents', 'storage', 'location'] as $type) {
            mkdir($user_dir . $type . '/', 0755, true);
        }

        http_response_code(201);
        echo json_encode([
            'success' => true,
            'message' => 'Signup successful',
            'user_id' => $user_id,
            'token' => $jwt
        ]);
    } catch (PDOException $e) {
        http_response_code(400);
        echo json_encode(['error' => 'Email already exists']);
    }
}
?>