<?php
// api/send_message.php - Send text/voice/file message
header('Content-Type: application/json');
require_once '../config.php';

$data = json_decode(file_get_contents('php://input'), true);
$token = $data['token'] ?? '';
$user_id = filter_var($data['user_id'] ?? 0, FILTER_VALIDATE_INT);
$other_user_id = filter_var($data['other_user_id'] ?? 0, FILTER_VALIDATE_INT);
$type = $data['type'] ?? 'text';
$content = $data['content'] ?? '';

if (!$user_id || !$token || !$other_user_id || !in_array($type, ['text', 'voice', 'file'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid input']);
    exit;
}

$pdo = getDB();
checkRateLimit($pdo, $user_id, 'send_message');

if (verifyJWT($token) !== $user_id) {
    http_response_code(401);
    echo json_encode(['error' => 'Invalid token']);
    exit;
}

// Find or create conversation
$stmt = $pdo->prepare("
    SELECT id FROM conversations 
    WHERE (user1_id = :user1 AND user2_id = :user2) OR (user1_id = :user2 AND user2_id = :user1)
");
$stmt->execute(['user1' => $user_id, 'user2' => $other_user_id]);
$convo = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$convo) {
    $stmt = $pdo->prepare("INSERT INTO conversations (user1_id, user2_id, created_at) VALUES (:user1, :user2, NOW())");
    $stmt->execute(['user1' => $user_id, 'user2' => $other_user_id]);
    $convo_id = $pdo->lastInsertId();
} else {
    $convo_id = $convo['id'];
}

$file_path = '';
if ($type === 'file' || $type === 'voice') {
    if (isset($_FILES['file'])) {
        $file = $_FILES['file'];
        $upload_dir = CHAT_UPLOAD_DIR . $convo_id . '/';
        if (!file_exists($upload_dir)) mkdir($upload_dir, 0755, true);

        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        $allowed = $type === 'voice' ? ['mp3', 'wav', 'm4a'] : ['jpg', 'jpeg', 'png', 'pdf', 'doc', 'docx'];
        if (!in_array($ext, $allowed)) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid file type']);
            exit;
        }
        if ($file['size'] > 10 * 1024 * 1024) { // 10MB limit
            http_response_code(400);
            echo json_encode(['error' => 'File too large']);
            exit;
        }

        $file_name = time() . '_' . basename($file['name']); // Prevent overwrites
        $target = $upload_dir . $file_name;
        if (move_uploaded_file($file['tmp_name'], $target)) {
            $file_path = '/uploads/chats/' . $convo_id . '/' . $file_name;
            $content = $file_path;
        } else {
            http_response_code(500);
            echo json_encode(['error' => 'File upload failed']);
            exit;
        }
    } else {
        http_response_code(400);
        echo json_encode(['error' => 'File required for type ' . $type]);
        exit;
    }
} elseif ($type === 'text' && empty($content)) {
    http_response_code(400);
    echo json_encode(['error' => 'Text content required']);
    exit;
}

$stmt = $pdo->prepare("INSERT INTO messages (conversation_id, sender_id, type, content, timestamp) VALUES (:convo_id, :sender_id, :type, :content, NOW())");
$stmt->execute(['convo_id' => $convo_id, 'sender_id' => $user_id, 'type' => $type, 'content' => $content]);

$stmt = $pdo->prepare("UPDATE conversations SET last_message_at = NOW() WHERE id = :id");
$stmt->execute(['id' => $convo_id]);

http_response_code(201);
echo json_encode(['success' => true, 'message_id' => $pdo->lastInsertId()]);
?>