<?php
// Set appropriate limits for file uploads
ini_set('memory_limit', '512M');
ini_set('max_execution_time', 300); // 5 minutes
ini_set('upload_max_filesize', '100M');
ini_set('post_max_size', '100M');

require_once '../config/database.php';

handleCors();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(['error' => 'Method not allowed'], 405);
}

try {
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        throw new Exception('Database connection failed');
    }
    
    // Get user info
    $user_email = $_POST['user_email'] ?? null;
    $backup_type = $_POST['backup_type'] ?? null; // contacts, files, images, videos, documents, audio
    
    if (!$user_email) {
        sendResponse(['error' => 'User email is required'], 400);
    }
    
    if (!$backup_type) {
        sendResponse(['error' => 'Backup type is required'], 400);
    }
    
    // Get user ID
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$user_email]);
    $user = $stmt->fetch();
    
    if (!$user) {
        sendResponse(['error' => 'User not found'], 404);
    }
    
    $user_id = $user['id'];
    $user_folder = createUserFolder($user_email);
    
    // Handle different backup types
    switch ($backup_type) {
        case 'contacts':
            handleContactsBackup($pdo, $user_id, $user_folder);
            break;
            
        case 'files':
        case 'images':
        case 'videos':
        case 'documents':
        case 'audio':
        case 'archives':
        case 'applications':
            handleFilesBackup($pdo, $user_id, $user_folder, $backup_type);
            break;
            
        case 'sync_status':
            handleSyncStatusUpdate($pdo, $user_id);
            break;
            
        default:
            sendResponse(['error' => 'Invalid backup type'], 400);
    }
    
} catch(Exception $e) {
    logError('Backup API Error: ' . $e->getMessage());
    sendResponse(['error' => 'Internal server error: ' . $e->getMessage()], 500);
}

function handleContactsBackup($pdo, $user_id, $user_folder) {
    // Get contacts JSON data
    $contacts_data = $_POST['contacts_data'] ?? null;
    
    if (!$contacts_data) {
        sendResponse(['error' => 'Contacts data is required'], 400);
    }
    
    $contacts = json_decode($contacts_data, true);
    
    if (!$contacts || !is_array($contacts)) {
        sendResponse(['error' => 'Invalid contacts data format'], 400);
    }
    
    $inserted_count = 0;
    $updated_count = 0;
    
    foreach ($contacts as $contact) {
        $contact_name = $contact['name'] ?? '';
        $phone_number = $contact['phone'] ?? '';
        $email = $contact['email'] ?? '';
        $contact_json = json_encode($contact);
        
        // Check if contact already exists
        $stmt = $pdo->prepare("SELECT id FROM contacts WHERE user_id = ? AND phone_number = ?");
        $stmt->execute([$user_id, $phone_number]);
        $existing_contact = $stmt->fetch();
        
        if ($existing_contact) {
            // Update existing contact
            $stmt = $pdo->prepare("
                UPDATE contacts 
                SET contact_name = ?, email = ?, contact_data = ?, is_synced = TRUE, updated_at = NOW() 
                WHERE id = ?
            ");
            $stmt->execute([$contact_name, $email, $contact_json, $existing_contact['id']]);
            $updated_count++;
        } else {
            // Insert new contact
            $stmt = $pdo->prepare("
                INSERT INTO contacts (user_id, contact_name, phone_number, email, contact_data, is_synced, created_at) 
                VALUES (?, ?, ?, ?, ?, TRUE, NOW())
            ");
            $stmt->execute([$user_id, $contact_name, $phone_number, $email, $contact_json]);
            $inserted_count++;
        }
    }
    
    // Save contacts as JSON file
    $contacts_file = $user_folder . '/contacts/contacts_backup_' . date('Y-m-d_H-i-s') . '.json';
    file_put_contents($contacts_file, $contacts_data);
    
    // Update sync status
    updateSyncStatus($pdo, $user_id, 'contacts', count($contacts), $inserted_count + $updated_count);
    
    sendResponse([
        'success' => true,
        'message' => 'Contacts backup completed',
        'inserted' => $inserted_count,
        'updated' => $updated_count,
        'total' => count($contacts),
        'backup_file' => $contacts_file
    ]);
}

function handleFilesBackup($pdo, $user_id, $user_folder, $backup_type) {
    $uploaded_files = [];
    $total_files = 0;
    $success_count = 0;
    $duplicate_count = 0;
    
    if (!isset($_FILES) || empty($_FILES)) {
        error_log("Backup upload failed - No files received. POST data: " . print_r($_POST, true));
        error_log("PHP upload limits - max_file_size: " . ini_get('upload_max_filesize') . 
                  ", post_max_size: " . ini_get('post_max_size') . 
                  ", max_input_time: " . ini_get('max_input_time'));
        sendResponse(['error' => 'No files to upload'], 400);
    }
    
    error_log("Received " . count($_FILES) . " files for backup_type: $backup_type");
    
    // Create backup subfolder
    $backup_path = $user_folder . '/backup/' . $backup_type;
    if (!file_exists($backup_path)) {
        mkdir($backup_path, 0755, true);
    }
    
    foreach ($_FILES as $field_name => $file) {
        if ($file['error'] !== UPLOAD_ERR_OK) {
            $error_message = getUploadErrorMessage($file['error']);
            error_log("File upload error for $field_name: $error_message (file: {$file['name']}, size: {$file['size']})");
            continue;
        }
        
        $total_files++;
        
        $file_name = $file['name'];
        $file_tmp = $file['tmp_name'];
        $file_size = $file['size'];
        $file_type = $file['type'];
        $original_path = $_POST['original_path_' . $field_name] ?? '';
        
        // Determine file type
        $file_extension = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
        $db_file_type = determineFileType($file_extension, $backup_type);
        
        // Create unique filename
        $unique_filename = time() . '_' . uniqid() . '_' . $file_name;
        $file_path = $backup_path . '/' . $unique_filename;
        
        if (move_uploaded_file($file_tmp, $file_path)) {
            // Calculate file hash
            $file_hash = calculateFileHash($file_path);
            
            // Check for duplicates
            $stmt = $pdo->prepare("SELECT id FROM files WHERE user_id = ? AND file_hash = ?");
            $stmt->execute([$user_id, $file_hash]);
            $existing_file = $stmt->fetch();
            
            if ($existing_file) {
                unlink($file_path);
                $duplicate_count++;
                $uploaded_files[] = [
                    'file_name' => $file_name,
                    'status' => 'duplicate',
                    'file_id' => $existing_file['id']
                ];
            } else {
                // Insert file record
                $stmt = $pdo->prepare("
                    INSERT INTO files (user_id, file_name, file_path, file_type, file_size, file_hash, mime_type, category, original_path, upload_status, upload_progress, created_at) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, 'backup', ?, 'completed', 100, NOW())
                ");
                $stmt->execute([$user_id, $file_name, $file_path, $db_file_type, $file_size, $file_hash, $file_type, $original_path]);
                
                $file_id = $pdo->lastInsertId();
                $success_count++;
                
                $uploaded_files[] = [
                    'file_name' => $file_name,
                    'status' => 'uploaded',
                    'file_id' => $file_id,
                    'file_size' => $file_size
                ];
            }
        } else {
            $uploaded_files[] = [
                'file_name' => $file_name,
                'status' => 'failed',
                'error' => 'Upload failed'
            ];
        }
    }
    
    // Update sync status
    updateSyncStatus($pdo, $user_id, $backup_type, $total_files, $success_count);
    
    sendResponse([
        'success' => true,
        'message' => ucfirst($backup_type) . ' backup completed',
        'total_files' => $total_files,
        'uploaded' => $success_count,
        'duplicates' => $duplicate_count,
        'files' => $uploaded_files
    ]);
}

function handleSyncStatusUpdate($pdo, $user_id) {
    // Get sync status for all types
    $stmt = $pdo->prepare("
        SELECT sync_type, total_items, synced_items, failed_items, sync_progress, is_completed, last_sync_timestamp 
        FROM sync_status 
        WHERE user_id = ?
    ");
    $stmt->execute([$user_id]);
    $sync_statuses = $stmt->fetchAll();
    
    sendResponse([
        'success' => true,
        'sync_status' => $sync_statuses
    ]);
}

function determineFileType($extension, $backup_type) {
    $image_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'bmp', 'tiff', 'heic', 'heif'];
    $video_extensions = ['mp4', 'avi', 'mov', 'mkv', 'webm', '3gp', 'flv', 'wmv', 'm4v', 'mpg', 'mpeg'];
    $audio_extensions = ['mp3', 'wav', 'aac', 'm4a', 'ogg', 'wma', 'flac', 'opus', 'amr'];
    $document_extensions = ['pdf', 'doc', 'docx', 'txt', 'xls', 'xlsx', 'ppt', 'pptx', 'rtf', 'odt', 'ods', 'csv'];
    $archive_extensions = ['zip', 'rar', '7z', 'tar', 'gz', 'bz2'];
    $app_extensions = ['apk'];
    
    if (in_array($extension, $image_extensions)) {
        return 'image';
    } elseif (in_array($extension, $video_extensions)) {
        return 'video';
    } elseif (in_array($extension, $audio_extensions)) {
        return 'audio';
    } elseif (in_array($extension, $document_extensions)) {
        return 'document';
    } elseif (in_array($extension, $archive_extensions)) {
        return 'archive';
    } elseif (in_array($extension, $app_extensions)) {
        return 'application';
    } else {
        return 'other';
    }
}

function getUploadErrorMessage($error_code) {
    switch ($error_code) {
        case UPLOAD_ERR_INI_SIZE:
            return 'File exceeds upload_max_filesize directive';
        case UPLOAD_ERR_FORM_SIZE:
            return 'File exceeds MAX_FILE_SIZE directive';
        case UPLOAD_ERR_PARTIAL:
            return 'File was only partially uploaded';
        case UPLOAD_ERR_NO_FILE:
            return 'No file was uploaded';
        case UPLOAD_ERR_NO_TMP_DIR:
            return 'Missing a temporary folder';
        case UPLOAD_ERR_CANT_WRITE:
            return 'Failed to write file to disk';
        case UPLOAD_ERR_EXTENSION:
            return 'File upload stopped by extension';
        default:
            return 'Unknown upload error';
    }
}

function updateSyncStatus($pdo, $user_id, $sync_type, $total_items, $synced_items) {
    $failed_items = $total_items - $synced_items;
    $sync_progress = $total_items > 0 ? ($synced_items / $total_items) * 100 : 0;
    $is_completed = $synced_items >= $total_items;
    
    $stmt = $pdo->prepare("
        UPDATE sync_status 
        SET total_items = ?, synced_items = ?, failed_items = ?, sync_progress = ?, is_completed = ?, last_sync_timestamp = NOW() 
        WHERE user_id = ? AND sync_type = ?
    ");
    $stmt->execute([$total_items, $synced_items, $failed_items, $sync_progress, $is_completed, $user_id, $sync_type]);
}
?>