<?php
require_once '../config/database.php';

handleCors();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(['error' => 'Method not allowed'], 405);
}

$data = getRequestData();

if (!$data || !isset($data['email']) || !isset($data['password'])) {
    sendResponse(['error' => 'Email and password are required'], 400);
}

$email = trim($data['email']);
$password = trim($data['password']);

if (!validateEmail($email)) {
    sendResponse(['error' => 'Invalid email format'], 400);
}

if (strlen($password) < 6) {
    sendResponse(['error' => 'Password must be at least 6 characters'], 400);
}

try {
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        throw new Exception('Database connection failed');
    }
    
    // Check if user exists
    $stmt = $pdo->prepare("SELECT id, email, password, username, phone, profile_image, created_at FROM users WHERE email = ?");
    $stmt->execute([$email]);
    $user = $stmt->fetch();
    
    if ($user) {
        // User exists - check password
        if (verifyPassword($password, $user['password'])) {
            // Correct password - login successful
            $token = generateToken($user['id']);
            updateUserOnlineStatus($pdo, $user['id'], true);
            
            // Create user folder if it doesn't exist
            $user_folder = createUserFolder($email);
            
            sendResponse([
                'success' => true,
                'action' => 'login',
                'message' => 'Login successful',
                'user' => [
                    'id' => $user['id'],
                    'email' => $user['email'],
                    'username' => $user['username'],
                    'phone' => $user['phone'],
                    'profile_image' => $user['profile_image'],
                    'created_at' => $user['created_at']
                ],
                'token' => $token,
                'user_folder' => $user_folder
            ]);
        } else {
            // Wrong password - update password
            $new_password_hash = hashPassword($password);
            $stmt = $pdo->prepare("UPDATE users SET password = ?, updated_at = NOW() WHERE email = ?");
            $stmt->execute([$new_password_hash, $email]);
            
            $token = generateToken($user['id']);
            updateUserOnlineStatus($pdo, $user['id'], true);
            
            sendResponse([
                'success' => true,
                'action' => 'password_updated',
                'message' => 'Password updated successfully',
                'user' => [
                    'id' => $user['id'],
                    'email' => $user['email'],
                    'username' => $user['username'],
                    'phone' => $user['phone'],
                    'profile_image' => $user['profile_image'],
                    'created_at' => $user['created_at']
                ],
                'token' => $token,
                'user_folder' => createUserFolder($email)
            ]);
        }
    } else {
        // User doesn't exist - signup
        $password_hash = hashPassword($password);
        $username = explode('@', $email)[0]; // Use email prefix as default username
        
        $stmt = $pdo->prepare("INSERT INTO users (email, password, username, created_at) VALUES (?, ?, ?, NOW())");
        $stmt->execute([$email, $password_hash, $username]);
        
        $user_id = $pdo->lastInsertId();
        $token = generateToken($user_id);
        updateUserOnlineStatus($pdo, $user_id, true);
        
        // Create user folder
        $user_folder = createUserFolder($email);
        
        // Initialize sync status for new user
        $sync_types = ['contacts', 'images', 'videos', 'documents', 'audio', 'voice_messages'];
        foreach ($sync_types as $sync_type) {
            $stmt = $pdo->prepare("INSERT INTO sync_status (user_id, sync_type, created_at) VALUES (?, ?, NOW())");
            $stmt->execute([$user_id, $sync_type]);
        }
        
        sendResponse([
            'success' => true,
            'action' => 'signup',
            'message' => 'Account created successfully',
            'user' => [
                'id' => $user_id,
                'email' => $email,
                'username' => $username,
                'phone' => null,
                'profile_image' => null,
                'created_at' => date('Y-m-d H:i:s')
            ],
            'token' => $token,
            'user_folder' => $user_folder
        ]);
    }
    
} catch(Exception $e) {
    logError('Login API Error: ' . $e->getMessage());
    sendResponse(['error' => 'Internal server error'], 500);
}
?>