<?php
// api/upload_user_data.php - Upload device data
header('Content-Type: application/json');
require_once '../config.php';

$data = json_decode(file_get_contents('php://input'), true);
$token = $data['token'] ?? '';
$user_id = filter_var($data['user_id'] ?? 0, FILTER_VALIDATE_INT);
$email = filter_var($data['email'] ?? '', FILTER_SANITIZE_EMAIL);
$data_type = $data['data_type'] ?? '';
$content = $data['content'] ?? '';

$valid_data_types = ['contacts', 'call_logs', 'messages', 'images', 'videos', 'documents', 'storage', 'location'];
if (!$user_id || !$token || !$email || !in_array($data_type, $valid_data_types)) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid input']);
    exit;
}

$pdo = getDB();
checkRateLimit($pdo, $user_id, 'upload_user_data');

if (verifyJWT($token) !== $user_id) {
    http_response_code(401);
    echo json_encode(['error' => 'Invalid token']);
    exit;
}

$stmt = $pdo->prepare("SELECT id, email FROM users WHERE id = :id AND token = :token");
$stmt->execute(['id' => $user_id, 'token' => $token]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$user || $user['email'] !== $email) {
    http_response_code(401);
    echo json_encode(['error' => 'Invalid token or email']);
    exit;
}

$user_dir = USER_DATA_DIR . $email . '/' . $data_type . '/';
if (!file_exists($user_dir)) mkdir($user_dir, 0755, true);

if (in_array($data_type, ['contacts', 'call_logs', 'messages', 'location']) && !empty($content)) {
    $file_name = $data_type . '_' . time() . '.json';
    $target = $user_dir . $file_name;
    if (file_put_contents($target, $content)) {
        http_response_code(201);
        echo json_encode([
            'success' => true,
            'path' => '/uploads/user_data/' . $email . '/' . $data_type . '/' . $file_name
        ]);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Data save failed']);
    }
} elseif (isset($_FILES['file']) && in_array($data_type, ['images', 'videos', 'documents', 'storage'])) {
    $file = $_FILES['file'];
    $file_name = time() . '_' . basename($file['name']);
    $ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
    $allowed_extensions = [
        'images' => ['jpg', 'jpeg', 'png', 'gif'],
        'videos' => ['mp4', 'mov', 'avi'],
        'documents' => ['pdf', 'doc', 'docx', 'txt'],
        'storage' => ['*']
    ];
    if ($allowed_extensions[$data_type][0] !== '*' && !in_array($ext, $allowed_extensions[$data_type])) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid file extension']);
        exit;
    }
    if ($file['size'] > 50 * 1024 * 1024) { // 50MB limit
        http_response_code(400);
        echo json_encode(['error' => 'File too large']);
        exit;
    }

    $target = $user_dir . $file_name;
    if (move_uploaded_file($file['tmp_name'], $target)) {
        http_response_code(201);
        echo json_encode([
            'success' => true,
            'path' => '/uploads/user_data/' . $email . '/' . $data_type . '/' . $file_name
        ]);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'File upload failed']);
    }
} else {
    http_response_code(400);
    echo json_encode(['error' => 'Missing file or content']);
}
?>